--[[
Copyright (C) GtX (Andy), 2021

Author: GtX | Andy
Date: 14.04.2021
Revision: FS25-01

Contact:
https://forum.giants-software.com
https://github.com/GtX-Andy

Important:
Not to be added to any mods / maps or modified from its current release form.
No modifications may be made to this script, including conversion to other game versions without written permission from GtX | Andy
Copying or removing any part of this code for external use without written permission from GtX | Andy is prohibited.

Darf nicht zu Mods / Maps hinzugefügt oder von der aktuellen Release-Form geändert werden.
Ohne schriftliche Genehmigung von GtX | Andy dürfen keine Änderungen an diesem Skript vorgenommen werden, einschließlich der Konvertierung in andere Spielversionen
Das Kopieren oder Entfernen irgendeines Teils dieses Codes zur externen Verwendung ohne schriftliche Genehmigung von GtX | Andy ist verboten.
]]


TransferOwnershipScreen = {}

TransferOwnershipScreen.MOD_NAME = g_currentModName
TransferOwnershipScreen.MOD_DIRECTORY = g_currentModDirectory

local TransferOwnershipScreen_mt = Class(TransferOwnershipScreen, ScreenElement)

function TransferOwnershipScreen.show(transferPoint, headerText)
	local controller = g_transferOwnershipScreen
	
	if controller == nil then
		controller = TransferOwnershipScreen.register()
		g_transferOwnershipScreen = controller
	end

	controller:setTransferPoint(transferPoint)
	controller:setHeaderText(headerText)

	g_gui:showGui("TransferOwnershipScreen")
end

function TransferOwnershipScreen.register()
    local controller = TransferOwnershipScreen.new()

	g_gui:loadGui(TransferOwnershipScreen.MOD_DIRECTORY .. "gui/TransferOwnershipScreen.xml", "TransferOwnershipScreen", controller)

    return controller
end

function TransferOwnershipScreen.new(target, customMt)
    local self = ScreenElement.new(target, customMt or TransferOwnershipScreen_mt)

    self.defaultHeaderText = TransferOwnershipScreen.getDefaultHeaderText()
	self.listItems = {}

    return self
end

function TransferOwnershipScreen.createFromExistingGui(gui, guiName)
	local controller = TransferOwnershipScreen.register()

	controller:setTransferPoint(gui.transferPoint)
	g_gui:showGui("TransferOwnershipScreen")
	
	g_transferOwnershipScreen = controller

	return controller
end

function TransferOwnershipScreen:onOpen()
    TransferOwnershipScreen:superClass().onOpen(self)

	self:updateBalance()

    g_messageCenter:subscribe(MessageType.TRANSFER_OWNERSHIP_UPDATE_OBJECTS, self.updateObjects, self)
	g_messageCenter:subscribe(MessageType.MONEY_CHANGED, self.updateBalance, self)
end

function TransferOwnershipScreen:onClose()
    TransferOwnershipScreen:superClass().onClose(self)

	g_messageCenter:unsubscribeAll(self)

	if not g_gui.currentlyReloading then
		self:removeTransferPoint(self.transferPoint, true)
	end
end

function TransferOwnershipScreen:update(dt)
    TransferOwnershipScreen:superClass().update(self, dt)

    if self.selectedItem ~= nil then
        if self.selectedItem.vehicle ~= nil then
            if self.selectedItem.vehicle.isDeleted then
                self:setObjects()
            end
        elseif self.selectedItem.objects ~= nil then
            local isBales = self.selectedItem.isBales

            for _, object in ipairs (self.selectedItem.objects) do
                if isBales then
                    if not entityExists(object.nodeId or 0) then
                        self:setObjects()

                        break
                    end
                elseif object.isDeleted then
                    self:setObjects()

                    break
                end
            end
        end
    end
end

function TransferOwnershipScreen:getNumberOfSections()
	return 1
end

function TransferOwnershipScreen:getNumberOfItemsInSection(list, section)
	return #self.listItems
end

function TransferOwnershipScreen:onListSelectionChanged(list, section, index)
    self.selectedItem = self.listItems[index]
	self.screenInfo:setVisible(self.selectedItem == nil)
end

function TransferOwnershipScreen:populateCellForItemInSection(list, section, index, cell)
	local item = self.listItems[index]
	local infoBoxVisible = false
	
	if item ~= nil then
		cell:getAttribute("icon"):setImageFilename(item.imageFilename or "data/store/store_empty.dds")
		cell:getAttribute("name"):setText(item.name or "Unknown")
	
		cell:getAttribute("typeNameText"):setText(item.typeName or "...")
	
		if item.vehicle ~= nil then
			local vehicle = item.vehicle
			local minutes = vehicle:getOperatingTime() / 60000
			local hours = math.floor(minutes / 60)
	
			cell:getAttribute("countTitle"):setLocaKey("ui_operatingHours:")
			cell:getAttribute("countText"):setText(string.format(g_i18n:getText("shop_operatingTime"), hours, math.floor((minutes - hours * 60) / 6) * 10))

			cell:getAttribute("fillLevelTitle"):setLocaKey("ui_licensePlate:")
			cell:getAttribute("fillLevelText"):setText(LicensePlates.getSpecValuePlateText(nil, vehicle) or "-")
	
			if vehicle.propertyState == VehiclePropertyState.OWNED then
				local sellPrice = math.floor(vehicle:getSellPrice() * EconomyManager.DIRECT_SELL_MULTIPLIER)
	
				cell:getAttribute("sellValueText"):setText(g_i18n:formatMoney(math.min(sellPrice, vehicle:getPrice())))
			elseif vehicle.propertyState == VehiclePropertyState.LEASED then
				cell:getAttribute("sellValueText"):setText(g_i18n:getText("ui_lease")) -- Future use
			end
			
			infoBoxVisible = true
		elseif item.objects ~= nil then
			local numObjects = #item.objects
			local fillLevel, highestSellPrice = 0, 0
	
			cell:getAttribute("countTitle"):setLocaKey("infohud_numAnimals:")
			cell:getAttribute("fillLevelTitle"):setLocaKey("info_fillLevel:")
	
			if numObjects > 0 then
				cell:getAttribute("countText"):setText(tostring(numObjects))
	
				if item.isPallets then
					for _, object in ipairs (item.objects) do
						fillLevel = fillLevel + (object:getFillUnitFillLevel(object.spec_pallet.fillUnitIndex) or 0)
					end
				elseif item.isBales then
					for _, object in ipairs (item.objects) do
						fillLevel = fillLevel + (object:getFillLevel() or 0)
					end
				end
	
				local fillTypeIndex = item.fillTypeIndex
	
				if fillTypeIndex ~= nil then
					for _, unloadingStation in pairs(g_currentMission.storageSystem:getUnloadingStations()) do
						if unloadingStation.owningPlaceable ~= nil and unloadingStation.isSellingPoint and unloadingStation.acceptedFillTypes[fillTypeIndex] then
							local price = unloadingStation:getEffectiveFillTypePrice(fillTypeIndex)
			
							if price > 0 then
								highestSellPrice = math.max(highestSellPrice, price)
							end
						end
					end
				end
	
				infoBoxVisible = true
			end
			
			cell:getAttribute("fillLevelText"):setText(g_i18n:formatVolume(fillLevel, 0, item.unitText))
			cell:getAttribute("sellValueText"):setText(g_i18n:formatMoney(fillLevel * highestSellPrice))
		end
	else
		cell:getAttribute("icon"):setImageFilename("data/store/store_empty.dds")
		cell:getAttribute("name"):setText("Unknown")
	end

	cell:getAttribute("infoBox"):setVisible(infoBoxVisible)
end

function TransferOwnershipScreen:removeTransferPoint(transferPoint, force)
    if self.transferPoint == transferPoint or force then
		self.transferPoint = nil
		self:setObjects()
	end
end

function TransferOwnershipScreen:setTransferPoint(transferPoint)
    if transferPoint == nil or transferPoint.determineCurrentObjects == nil then
        return false
    end

    self.transferPoint = transferPoint
    self:setObjects()

    return true
end

function TransferOwnershipScreen:setHeaderText(text)
	if self.headerText ~= nil then
		self.headerText:setText(text or self.defaultHeaderText)
	end
end

function TransferOwnershipScreen:setObjects()
    local maxObjectsPerGroup = 2 ^ TransferOwnershipEvent.SEND_NUM_BITS - 1

	for i = #self.listItems, 1, -1 do
		self.listItems[i] = nil
	end

	if self.transferPoint and self.transferPoint.determineCurrentObjects ~= nil then
		local farmId = g_currentMission:getFarmId()
		local vehicles, bales = self.transferPoint:determineCurrentObjects()

		if vehicles ~= nil and #vehicles > 0 then
			local palletGroups, bigBagText, palletText

			for _, vehicle in ipairs (vehicles) do
                if vehicle.ownerFarmId == farmId then
                    if vehicle.isPallet and vehicle.spec_pallet ~= nil then
						local fillUnit = vehicle:getFillUnitByIndex(vehicle.spec_pallet.fillUnitIndex)
                        local fillTypeIndex = vehicle:getFillUnitFillType(vehicle.spec_pallet.fillUnitIndex)
                        local fillType = g_fillTypeManager:getFillTypeByIndex(fillTypeIndex)

                        if fillUnit ~= nil and fillType ~= nil then
                            local isBigBag = SpecializationUtil.hasSpecialization(BigBag, vehicle.specializations)
                            local typeKey, nameI18N = isBigBag and "BIGBAG" or "PALLET", nil

                            if vehicle.spec_treeSaplingPallet ~= nil then
                                local treeTypeDesc = vehicle.spec_treeSaplingPallet.treeTypeDesc

                                if treeTypeDesc ~= nil then
                                    typeKey = treeTypeDesc.name
                                    nameI18N = treeTypeDesc.nameI18N
                                end
                            end
							
							if palletGroups == nil then
								palletGroups = {}
								bigBagText = g_i18n:getText("shopItem_bigBag")
								palletText = g_i18n:getText("infohud_pallet")
							end

                            local key = string.format("%d_%s", fillTypeIndex, typeKey)

                            if palletGroups[key] == nil then
                                local name = fillType.title

                                if nameI18N ~= nil then
                                    name = string.format("%s (%s)", name, g_i18n:getText(nameI18N, vehicle.customEnvironment))
                                end

                                palletGroups[key] = {
                                    imageFilename = fillType.hudOverlayFilename,
                                    fillTypeIndex = fillTypeIndex,
                                    name = name,
                                    unitText = fillUnit.unitText,
                                    typeName = isBigBag and bigBagText or palletText,
                                    objects = {},
                                    isBigBags = isBigBag,
                                    isPallets = true,
									sortId = 2
                                }

                                table.insert(self.listItems, palletGroups[key])
                            end

                            if #palletGroups[key].objects < maxObjectsPerGroup then
                                table.insert(palletGroups[key].objects, vehicle)
                            end
                        end
					else
						local brand = g_brandManager:getBrandByIndex(vehicle:getBrand())

						table.insert(self.listItems, {
                            imageFilename = vehicle:getImageFilename(),
                            name = vehicle:getName(),
                            typeName = brand ~= nil and brand.title or "N/A",
                            vehicle = vehicle,
                            isVehicle = true,
							sortId = 1
                        })
					end
				end
			end
		end
	
		if bales ~= nil and #bales > 0 then
            local baleGroups, roundText, squareText, fermentingText

            for _, bale in ipairs (bales) do
                if bale.ownerFarmId == farmId then
                    local fillTypeIndex = bale:getFillType()
                    local fillType = g_fillTypeManager:getFillTypeByIndex(fillTypeIndex)

                    if fillType ~= nil then
                        if baleGroups == nil then
							baleGroups = {}

							roundText = g_i18n:getText("fillType_roundBale")
							squareText = g_i18n:getText("fillType_squareBale")
							fermentingText = g_i18n:getText("info_fermenting")
						end

						local key = string.format("%d_%s_%s", fillTypeIndex, bale.isRoundbale, bale.isFermenting)

                        if baleGroups[key] == nil then
                            local name = fillType.title

                            if bale.isFermenting then
                                name = string.format("%s (%s)", name, fermentingText)
                            end

                            baleGroups[key] = {
                                imageFilename = fillType.hudOverlayFilename,
                                fillTypeIndex = fillTypeIndex,
                                name = name,
                                typeName = bale.isRoundbale and roundText or squareText,
                                objects = {},
                                isBales = true,
								sortId = 3
                            }

                            table.insert(self.listItems, baleGroups[key])
                        end

                        if #baleGroups[key].objects < maxObjectsPerGroup then
                            table.insert(baleGroups[key].objects, bale)
                        end
                    end
                end
            end
        end
	end
	
	local numTriggerObjects = #self.listItems
	
	if numTriggerObjects > 0 then
		table.sort(self.listItems, TransferOwnershipScreen.sortTriggerObjects)
	end

	self.list:reloadData()

    if numTriggerObjects == 0 then
        self.selectedItem = nil
		self.screenInfo:setVisible(true)
    end
end

function TransferOwnershipScreen:updateObjects(transferPoint)
    if transferPoint and transferPoint == self.transferPoint then
        self:setObjects()

		return true
    end
	
	return false
end

function TransferOwnershipScreen:updateBalance()
	local balance = g_currentMission ~= nil and g_currentMission:getMoney() or 0

	self.farmBalanceText:setValue(balance)
	self.farmBalanceText:applyProfile(balance > 0 and "fs25_shopMoney" or "fs25_shopMoneyNeg")

	self.farmBalanceBox:invalidateLayout()
	self.farmBalanceBoxBg:setSize(self.farmBalanceBox.flowSizes[1] + 60 * g_pixelSizeScaledX)
end

function TransferOwnershipScreen:onClickBack(forceBack, usedMenuButton)
    self:changeScreen(nil)

    return true
end

function TransferOwnershipScreen:onClickTransfer()
    if not g_currentMission.missionDynamicInfo.isMultiplayer then
		local text = string.format("%s\n\n%s", g_i18n:getText("warning_actionNotAllowedNow"), g_i18n:getText("ui_joinMultiplayerGame"))

		InfoDialog.show(text, nil, nil, DialogElement.TYPE_INFO, g_i18n:getText("button_back"), InputAction.MENU_BACK)
		self:playSample(GuiSoundPlayer.SOUND_SAMPLES.ERROR)

        return true
    end

    local spectatorFarmId = FarmManager.SPECTATOR_FARM_ID or 0
	local maxFarmId = FarmManager.MAX_FARM_ID or 8
	local senderFarmId = g_currentMission:getFarmId() or spectatorFarmId

    if self.selectedItem ~= nil then
        local objectsToSend = {}

        if self.selectedItem.objects ~= nil then
            for _, object in ipairs (self.selectedItem.objects) do
                table.insert(objectsToSend, object)
            end
        elseif self.selectedItem.vehicle ~= nil then
            table.insert(objectsToSend, self.selectedItem.vehicle)
        end

        if #objectsToSend > 0 and senderFarmId ~= spectatorFarmId and senderFarmId <= maxFarmId then
            local farmNames = {}
            local farms = {}

            for _, farm in ipairs (g_farmManager:getFarms()) do
                if farm.farmId ~= senderFarmId and farm.farmId ~= spectatorFarmId and farm.farmId <= maxFarmId then
                    local name = farm.name

					if string.isNilOrWhitespace(name) then
						name = string.format("Unknown (%d)", farm.farmId)
					end

					table.insert(farmNames, name)
                    table.insert(farms, farm)
                end
            end

            if #farms > 0 then
                local callback = function (index)
                    if index > 0 then
                        local farm = farms[index]

                        if farm ~= nil and g_farmManager:getFarmById(farm.farmId) ~= nil then
                            if self:onConfirmTransfer(senderFarmId, farm.farmId, objectsToSend) then
                                self:playSample(GuiSoundPlayer.SOUND_SAMPLES.TRANSACTION)
                            end
                        end
                    end
                end

				OptionDialog.show(callback, g_i18n:getText("ui_transferOwnershipSelectFarm", TransferOwnershipScreen.MOD_NAME), self.headerText:getText(), farmNames, 1)
            else
				InfoDialog.show(g_i18n:getText("ui_noFarmsCreated"), nil, nil, DialogElement.TYPE_INFO, g_i18n:getText("button_back"), InputAction.MENU_BACK)
                self:playSample(GuiSoundPlayer.SOUND_SAMPLES.ERROR)
            end

            return false
        end
    end

    self:setObjects()

    return true
end

function TransferOwnershipScreen:onConfirmTransfer(senderFarmId, destinationFarmId, objects)
    if senderFarmId ~= nil and destinationFarmId ~= nil then
		local owningPlaceable = nil
	
		if self.transferPoint ~= nil then
			owningPlaceable = self.transferPoint.owningPlaceable
		end

		if g_server ~= nil then
			TransferOwnershipEvent.setObjectsOwnerFarmId(objects, senderFarmId, destinationFarmId, owningPlaceable)
	
			g_messageCenter:publish(MessageType.TRANSFER_OWNERSHIP_UPDATE_OBJECTS, nil) -- Only trigger 'shopController' update so send nil for 'transferPoint'
	
			self:setObjects()
		else
            g_client:getServerConnection():sendEvent(TransferOwnershipEvent.new(objects, destinationFarmId, owningPlaceable))
		end

		return true
	end

	return false
end

function TransferOwnershipScreen.sortTriggerObjects(a, b)
	if a.sortId == b.sortId then
		return a.name < b.name
	end
	
	return a.sortId < b.sortId
end

function TransferOwnershipScreen.getDefaultHeaderText()
	local transferOwnershipText = g_i18n:getText("ui_transferOwnershipTitle", TransferOwnershipScreen.MOD_NAME)

	if g_languageShort ~= "en" and transferOwnershipText == "Transfer Ownership" then
		transferOwnershipText = string.format(g_i18n:getText("button_mp_transferMoney_dialogTitle"), g_i18n:getText("ui_farm"))
	end

	return transferOwnershipText
end
